let state = {
    queue: [],
    results: [],
    totalPrompts: 0,
    currentIndex: 0,
    delay: 5,
    isRunning: false,
    isPaused: false,
    smartRetry: true
};

// Load state from storage on service worker startup
chrome.storage.local.get(['aiState'], (result) => {
    if (result.aiState) {
        state = result.aiState;
    }
});

// Broadcast state to popup and save to storage
function saveAndBroadcastState() {
    chrome.storage.local.set({ aiState: state });
    chrome.runtime.sendMessage({ type: "STATE_UPDATE", state }).catch(() => {
        // Popup might be closed, ignore error
    });
}

// Save completed or stopped queue results to history
function saveToHistory() {
    if (!state.results || state.results.length === 0) return;

    chrome.storage.local.get(['historyLogs'], (result) => {
        let logs = result.historyLogs || [];
        const newEntry = {
            id: Date.now(),
            date: new Date().toLocaleString(),
            totalPrompts: state.totalPrompts,
            completedPrompts: state.results.length,
            results: [...state.results]
        };
        logs.unshift(newEntry);
        if (logs.length > 50) logs = logs.slice(0, 50);
        chrome.storage.local.set({ historyLogs: logs });
    });
}

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === "GET_STATE") {
        sendResponse(state);
        return true;
    }

    if (message.type === "START") {
        state.queue = message.payload.prompts;
        state.results = [];
        state.totalPrompts = state.queue.length;
        state.currentIndex = 0;
        state.delay = message.payload.delay;
        if (message.payload.smartRetry !== undefined) {
            state.smartRetry = message.payload.smartRetry;
        }
        state.isRunning = true;
        state.isPaused = false;
        saveAndBroadcastState();
        processNext();
    }

    if (message.type === "TOGGLE_PAUSE") {
        if (state.isRunning || state.isPaused) {
            state.isPaused = !state.isPaused;
            state.isRunning = !state.isPaused;
            // Clear any pending alarms if paused
            if (state.isPaused) {
                chrome.alarms.clear("nextPromptAlarm");
            }
            saveAndBroadcastState();
            if (state.isRunning) {
                processNext();
            }
        }
    }

    if (message.type === "STOP") {
        if (state.results && state.results.length > 0) {
            saveToHistory();
        }
        state.queue = [];
        state.totalPrompts = 0;
        state.currentIndex = 0;
        state.isRunning = false;
        state.isPaused = false;
        chrome.alarms.clear("nextPromptAlarm");
        saveAndBroadcastState();
    }

    if (message.type === "PROMPT_DONE") {
        const promptText = state.queue[state.currentIndex];
        const responseText = message.response || "No response extracted.";
        const responseHtml = message.responseHtml || "<p>No response extracted.</p>";

        // Ensure state.results exists (for backward compatibility if migrating state)
        if (!state.results) state.results = [];
        state.results.push({ prompt: promptText, response: responseText, responseHtml: responseHtml });

        state.currentIndex++;
        saveAndBroadcastState();

        if (state.currentIndex >= state.totalPrompts) {
            state.isRunning = false;
            state.isPaused = false;
            saveToHistory();
            saveAndBroadcastState();

            // Trigger System Notification
            chrome.notifications.create({
                type: 'basic',
                iconUrl: 'logo.png',
                title: 'AI Auto-Pilot Complete',
                message: `Successfully processed ${state.totalPrompts} prompts!`,
                priority: 2
            });

            // Broadcast completion message explicitly to popup (if open)
            chrome.runtime.sendMessage({
                type: "QUEUE_COMPLETED",
                total: state.totalPrompts
            }).catch(() => {
                // Popup might be closed, ignore error
            });
        } else {
            // Schedule the next prompt using Alarms API instead of setTimeout
            // Alarms API takes minutes, so we convert delay (seconds) to minutes
            // If the delay is very short (< 1 min), alarms still triggers but it's more reliable than setTimeout for SW
            chrome.alarms.create("nextPromptAlarm", {
                delayInMinutes: state.delay / 60
            });
        }
    }

    if (message.type === "UPDATE_SETTINGS") {
        if (message.payload.smartRetry !== undefined) {
            state.smartRetry = message.payload.smartRetry;
            saveAndBroadcastState();
        }
    }

    if (message.type === "TAB_READY") {
        if (state.isRunning && !state.isPaused && state.smartRetry) {
            console.log("Tab ready signal received. Resuming processNext() for Smart Retry...");
            // Clear any pending alarms to avoid double execution
            chrome.alarms.clear("nextPromptAlarm");
            // Slight delay to ensure page is fully functional
            setTimeout(() => {
                processNext();
            }, 2000);
        }
    }
});

// Listen for alarms to wake up the service worker and process the next item
chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === "nextPromptAlarm") {
        processNext();
    }
});

function processNext() {
    if (!state.isRunning || state.isPaused) return;
    if (state.currentIndex >= state.totalPrompts) return;

    const prompt = state.queue[state.currentIndex];

    // Find active tab to execute prompt
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs.length === 0) {
            // If no active tab or window is minimized, query specifically for the AI domains
            chrome.tabs.query({
                url: [
                    "*://chatgpt.com/*",
                    "*://gemini.google.com/*",
                    "*://claude.ai/*",
                    "*://copilot.microsoft.com/*",
                    "*://*.perplexity.ai/*",
                    "*://chat.deepseek.com/*",
                    "*://duckduckgo.com/*",
                    "*://kimi.moonshot.cn/*",
                    "*://*.aifiesta.ai/*",
                    "*://*.lumio.ai/*",
                    "*://*.lumioai.tech/*",
                    "*://*.blaze.ai/*",
                    "*://you.com/*",
                    "*://grok.com/*",
                    "*://x.com/*",
                    "*://www.meta.ai/*"
                ]
            }, (aiTabs) => {
                if (aiTabs.length > 0) {
                    injectAndExecute(aiTabs[0].id, prompt);
                } else {
                    handleContentScriptError();
                }
            });
            return;
        }

        injectAndExecute(tabs[0].id, prompt);
    });
}

function injectAndExecute(activeTabId, prompt) {
    function sendMessageToTab() {
        chrome.tabs.sendMessage(activeTabId, {
            type: "EXECUTE_PROMPT",
            prompt: prompt,
            smartRetry: state.smartRetry
        }, (response) => {
            if (chrome.runtime.lastError) {
                console.log("Content script not responding. Attempting to inject...", chrome.runtime.lastError);
                // Inject script dynamically if Tab was opened before extension was installed
                chrome.scripting.executeScript({
                    target: { tabId: activeTabId },
                    files: ['content.js']
                }).then(() => {
                    // Wait briefly for content script to initialize then try again
                    setTimeout(() => {
                        chrome.tabs.sendMessage(activeTabId, {
                            type: "EXECUTE_PROMPT",
                            prompt: prompt,
                            smartRetry: state.smartRetry
                        }, () => {
                            if (chrome.runtime.lastError) {
                                console.error("Injection failed to resolve connection:", chrome.runtime.lastError);
                                handleContentScriptError();
                            }
                        });
                    }, 500);
                }).catch((err) => {
                    console.error("Failed to inject script:", err);
                    handleContentScriptError();
                });
            }
        });
    }

    sendMessageToTab();
}

function handleContentScriptError() {
    if (state.smartRetry) {
        console.warn("AI Auto-Pilot paused temporarily: Waiting for AI tab or internet. Smart Retry is active.");
        chrome.alarms.clear("nextPromptAlarm");
        // State remains isRunning=true. When a tab reloads, it will send TAB_READY and resume.
        saveAndBroadcastState();
    } else {
        state.isPaused = true;
        state.isRunning = false;
        chrome.alarms.clear("nextPromptAlarm");
        saveAndBroadcastState();
        console.warn("AI Auto-Pilot paused: Could not communicate with an active AI tab.");
    }
}
