// Platforms Supported Map
function getPlatform() {
    const url = window.location.href;
    if (url.includes("chatgpt.com")) return "chatgpt";
    if (url.includes("gemini.google.com")) return "gemini";
    if (url.includes("claude.ai")) return "claude";
    if (url.includes("copilot.microsoft.com")) return "copilot";
    if (url.includes("perplexity.ai")) return "perplexity";
    if (url.includes("deepseek.com")) return "deepseek";
    if (url.includes("duckduckgo.com") && url.includes("chat")) return "duckduckgo";
    if (url.includes("duck.ai")) return "duckduckgo";
    if (url.includes("duckduckgo.com")) return "duckduckgo";
    if (url.includes("kimi.moonshot.cn")) return "kimi";
    if (url.includes("aifiesta.ai") || url.includes("aifiesta.online")) return "aifiesta";
    if (url.includes("lumio.ai") || url.includes("lumioai.tech")) return "lumio";
    if (url.includes("blaze.ai") || url.includes("blaze.today")) return "blaze";
    if (url.includes("you.com")) return "youchat";
    if (url.includes("grok.com") || url.includes("x.com/i/grok")) return "grok";
    if (url.includes("meta.ai")) return "meta";
    return null;
}

const Selectors = {
    chatgpt: {
        input: '#prompt-textarea',
        submit: '[data-testid="send-button"]',
        stopGenerating: '[data-testid="stop-button"], [aria-label="Stop generating"]'
    },
    gemini: {
        input: 'div[role="textbox"]',
        submit: 'button[aria-label="Send message"]',
        stopGenerating: 'button[aria-label="Stop responding"]'
    },
    claude: {
        input: 'div[contenteditable="true"]',
        submit: 'button[aria-label="Send Message"]',
        stopGenerating: 'button[aria-label="Stop generating"]'
    },
    copilot: {
        input: '#searchbox, textarea, div[contenteditable="true"]',
        submit: 'button[aria-label="Submit"], button[title="Submit"]',
        stopGenerating: 'button[aria-label="Stop responding"]'
    },
    perplexity: {
        input: 'textarea, div[contenteditable="true"]',
        submit: 'button[aria-label="Submit"], button:has(svg)',
        stopGenerating: 'button[aria-label="Stop generating"]'
    },
    deepseek: {
        input: '#chat-input, textarea',
        submit: '.ds-button, button:has(svg)',
        stopGenerating: '.ds-button-stop'
    },
    duckduckgo: {
        input: 'textarea, input[type="text"], div[contenteditable="true"]',
        submit: 'button[type="submit"]',
        stopGenerating: 'button:contains("Stop")'
    },
    kimi: {
        input: 'div[contenteditable="true"], textarea',
        submit: 'button[data-testid="send-button"]',
        stopGenerating: 'button[data-testid="stop-button"]'
    },
    aifiesta: {
        input: 'textarea, div[role="textbox"], div[contenteditable="true"]',
        submit: 'button[type="submit"], button[aria-label="Send"]',
        stopGenerating: 'button:contains("Stop"), button[aria-label="Stop"]'
    },
    lumio: {
        input: 'textarea, div[role="textbox"], div[contenteditable="true"]',
        submit: 'button[type="submit"], button[aria-label="Send"]',
        stopGenerating: 'button:contains("Stop")'
    },
    blaze: {
        input: 'textarea, div[contenteditable="true"]',
        submit: 'button[type="submit"], button[aria-label="Send message"]',
        stopGenerating: 'button:contains("Stop")'
    },
    youchat: {
        input: 'textarea[placeholder*="Ask anything"], textarea, div[contenteditable="true"]',
        submit: 'button[title="Send message"], button[aria-label="Send"]',
        stopGenerating: 'button[aria-label="Stop generation"]'
    },
    grok: {
        input: 'textarea, div[role="textbox"], div[contenteditable="true"]',
        submit: 'button[aria-label="Grok something"]',
        stopGenerating: 'button:contains("Stop")'
    },
    meta: {
        input: 'textarea, div[role="textbox"], div[contenteditable="true"]',
        submit: 'div[role="button"][aria-label="Send"]',
        stopGenerating: 'div[role="button"][aria-label="Stop generating"]'
    }
};

function waitForElement(selector, timeoutMs = 5000) {
    return new Promise((resolve) => {
        const el = document.querySelector(selector);
        if (el) return resolve(el);

        const observer = new MutationObserver((mutations, obs) => {
            const el = document.querySelector(selector);
            if (el) {
                obs.disconnect();
                resolve(el);
            }
        });
        observer.observe(document.body, { childList: true, subtree: true });

        setTimeout(() => {
            observer.disconnect();
            resolve(null);
        }, timeoutMs);
    });
}

async function setTextToInput(platform, text) {
    const sel = Selectors[platform];
    const inputEl = await waitForElement(sel.input);
    if (!inputEl) return false;

    inputEl.focus();

    if (inputEl.tagName.toLowerCase() === 'textarea' || inputEl.tagName.toLowerCase() === 'input') {
        // Native setting
        inputEl.value = text;

        // React/Vue hack: React tracks the value setter, we have to bypass it
        const nativeInputValueSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, "value")?.set;
        const nativeTextAreaValueSetter = Object.getOwnPropertyDescriptor(window.HTMLTextAreaElement.prototype, "value")?.set;

        if (inputEl.tagName.toLowerCase() === 'textarea' && nativeTextAreaValueSetter) {
            nativeTextAreaValueSetter.call(inputEl, text);
        } else if (nativeInputValueSetter) {
            nativeInputValueSetter.call(inputEl, text);
        }

        inputEl.dispatchEvent(new Event('input', { bubbles: true }));
        inputEl.dispatchEvent(new Event('change', { bubbles: true }));
    } else {
        // ContentEditable (used by Gemini, Claude, etc)
        document.execCommand('selectAll', false, null);
        document.execCommand('insertText', false, text);
        inputEl.dispatchEvent(new Event('input', { bubbles: true }));
        inputEl.dispatchEvent(new Event('change', { bubbles: true }));
    }
    return true;
}

function clickSubmit(platform) {
    const sel = Selectors[platform];
    const inputEl = document.querySelector(sel.input);
    const btn = document.querySelector(sel.submit);

    let clicked = false;

    // Attempt 1: Click the button directly
    if (btn && !btn.disabled) {
        btn.click();
        clicked = true;
    }

    // Attempt 2: Dispatch Enter key event on the input field (Very common in modern React apps)
    if (inputEl) {
        const enterEvent = new KeyboardEvent('keydown', {
            key: 'Enter',
            code: 'Enter',
            keyCode: 13,
            which: 13,
            bubbles: true,
            cancelable: true
        });
        inputEl.dispatchEvent(enterEvent);
        clicked = true;
    }

    return clicked;
}

function isGenerating(platform) {
    const sel = Selectors[platform];
    if (!sel) return false;

    try {
        const stopBtn = document.querySelector(sel.stopGenerating);

        if (platform === 'chatgpt') {
            const sendBtn = document.querySelector(sel.submit);
            return !!stopBtn || (sendBtn && sendBtn.disabled);
        }

        if (platform === 'gemini') {
            return !!stopBtn;
        }

        if (platform === 'claude') {
            const hasStop = Array.from(document.querySelectorAll('button')).some(b =>
                (b.getAttribute('aria-label') && b.getAttribute('aria-label').includes('Stop generating')) ||
                (b.textContent && b.textContent.includes('Stop generating'))
            );
            return hasStop || !!stopBtn;
        }

        if (stopBtn) return true;

        const submitBtn = document.querySelector(sel.submit);
        if (submitBtn && submitBtn.disabled) return true;

    } catch (e) {
        console.warn(`Error checking if ${platform} is generating:`, e);
    }

    return false;
}

async function extractResponse(platform) {
    let text = "";
    let html = "";

    function cleanHtml(element) {
        if (!element) return "";
        // Clone to not affect actual DOM
        const clone = element.cloneNode(true);
        // Remove copy buttons, read aloud buttons, etc common in AI UIs
        const unwanted = clone.querySelectorAll('button, svg, .copy-button');
        unwanted.forEach(el => el.remove());

        // Process images so they render correctly when exported outside the platform
        const images = clone.querySelectorAll('img');
        images.forEach(img => {
            // Apply a default max-width for export layouts
            img.style.maxWidth = '100%';
            img.style.height = 'auto';
            img.style.display = 'block';
            img.style.margin = '10px 0';

            // Fix lazy-loaded images (e.g. data-src)
            let src = img.getAttribute('src') || '';
            const dataSrc = img.getAttribute('data-src');
            if (dataSrc) {
                src = dataSrc;
                img.setAttribute('src', dataSrc);
            }

            // Convert relative URLs to absolute URLs
            if (src && src.startsWith('/')) {
                img.setAttribute('src', window.location.origin + src);
            }

            // Remove attributes that interfere with static rendering
            img.removeAttribute('srcset');
            img.removeAttribute('loading');
        });

        return clone.innerHTML;
    }

    if (platform === 'chatgpt') {
        const elements = document.querySelectorAll('div[data-message-author-role="assistant"]');
        if (elements.length > 0) {
            const el = elements[elements.length - 1];
            text = el.innerText;
            html = cleanHtml(el);
        }
    } else if (platform === 'gemini') {
        const elements = document.querySelectorAll('.model-response-text, message-content, model-response');
        if (elements.length > 0) {
            const el = elements[elements.length - 1];
            text = el.innerText;
            html = cleanHtml(el);
        }
    } else if (platform === 'claude') {
        const elements = document.querySelectorAll('.font-claude-message');
        if (elements.length > 0) {
            const el = elements[elements.length - 1];
            text = el.innerText;
            html = cleanHtml(el);
        }
    } else {
        const outputBlocks = document.querySelectorAll('.prose, .markdown, .message, div[dir="auto"], div[data-message-author-role="assistant"], div[class*="assistant"]');
        if (outputBlocks.length > 0) {
            for (let i = outputBlocks.length - 1; i >= 0; i--) {
                const el = outputBlocks[i];
                if (el.innerText && el.innerText.trim().length > 0) {
                    text = el.innerText;
                    html = cleanHtml(el);
                    break;
                }
            }
        }

        if (!text) {
            const tryCommon = document.querySelectorAll('[class*="response"], [class*="answer"], [class*="message"]');
            if (tryCommon.length > 0) {
                const el = tryCommon[tryCommon.length - 1];
                text = el.innerText;
                html = cleanHtml(el);
            }
        }
    }

    if (!text && !html) {
        await new Promise(r => setTimeout(r, 500));
        return { text: "No response extracted or extraction selector is outdated.", html: "<p>No response extracted.</p>" };
    }

    return { text, html };
}

async function executePromptTask(prompt, smartRetry = true) {
    const platform = getPlatform();
    if (!platform) {
        console.warn("AI Auto-Pilot: Not a supported platform (ChatGPT, Gemini, Claude).");
        return;
    }

    // 1. Insert text
    const success = await setTextToInput(platform, prompt);
    if (!success) {
        console.error(`AI Auto-Pilot: Input box not found for selector: ${Selectors[platform].input}`);
        return;
    }

    // Focus out and wait a tiny bit to let JS frameworks register the text
    document.body.focus();
    setTimeout(() => {
        // 2. Click send
        const typed = clickSubmit(platform);
        if (!typed) {
            console.error("AI Auto-Pilot: Could not click submit button.");
            return;
        }

        // 3. Monitor generation completion
        // Wait for generation to actually start (give it ~2 seconds buffer)
        setTimeout(() => {
            const checkInterval = setInterval(async () => {
                // Check for network disconnects
                if (smartRetry && !navigator.onLine) {
                    console.warn("AI Auto-Pilot: Internet disconnected. Triggering reload for Smart Retry...");
                    clearInterval(checkInterval);
                    window.location.reload();
                    return;
                }

                if (!isGenerating(platform)) {
                    clearInterval(checkInterval);

                    // Give the UI a moment to fully render the final response text
                    await new Promise(r => setTimeout(r, 1000));

                    const extracted = await extractResponse(platform);

                    // 4. Send done message to background
                    chrome.runtime.sendMessage({
                        type: "PROMPT_DONE",
                        response: extracted.text,
                        responseHtml: extracted.html
                    });
                }
            }, 1500); // Check every 1.5 seconds to save resources
        }, 2500);
    }, 800);
}

// Ensure script is injected only once.
if (!window.__AIAUTOPILOT_INJECTED) {
    window.__AIAUTOPILOT_INJECTED = true;

    // Send signal to background that tab is ready (handles page reloads/crashes)
    chrome.runtime.sendMessage({ type: "TAB_READY" }).catch(() => { });

    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
        if (message.type === "EXECUTE_PROMPT") {
            // Send immediate response so background doesn't throw error
            sendResponse({ status: "received" });

            // Execute task
            executePromptTask(message.prompt, message.smartRetry);
        }
        return true; // Keep message channel open for async response if needed (though we respond immediately above)
    });

    console.log("AI Auto-Pilot Content Script Loaded!");
}
