document.addEventListener("DOMContentLoaded", () => {
    const resultsContainer = document.getElementById("results-container");
    const printBtn = document.getElementById("print-btn");
    const closeBtn = document.getElementById("close-btn");

    printBtn.addEventListener("click", () => {
        window.print();
    });

    closeBtn.addEventListener("click", () => {
        window.close();
    });

    // Fetch state and title
    chrome.runtime.sendMessage({ type: "GET_STATE" }, (state) => {
        chrome.storage.local.get("tempExportTitle", (data) => {
            const customTitle = data.tempExportTitle || "AI Auto-Pilot Export";
            document.title = customTitle; // Update browser tab title

            if (!state || !state.results || state.results.length === 0) {
                resultsContainer.innerHTML = "<p>No results found to export. Please run a prompt queue first.</p>";
                return;
            }

            renderResults(state.results, customTitle);

            // Auto-print prompt after a short delay to allow styling and fonts to render
            setTimeout(() => {
                window.print();
            }, 500);
        });
    });

    function renderResults(results, title) {
        resultsContainer.innerHTML = "";

        let header = document.createElement("div");
        header.className = "export-header print-only";
        header.innerHTML = `<h2>${escapeHTML(title)}</h2><p>Exported on: ${new Date().toLocaleString()}</p>`;
        resultsContainer.appendChild(header);

        // Generate Index Table Container
        let indexContainer = document.createElement("div");
        indexContainer.style.marginBottom = "30px";
        indexContainer.style.border = "1px solid var(--border-color)";
        indexContainer.style.padding = "15px";
        indexContainer.style.borderRadius = "8px";
        indexContainer.style.pageBreakAfter = "always";

        let indexRows = results.map((res, i) => {
            let promptPreview = res.prompt || `Response ${i + 1}`;
            if (promptPreview.length > 60) {
                promptPreview = promptPreview.substring(0, 60) + "...";
            }
            return `
              <tr>
                <td style="text-align: center;">${i + 1}</td>
                <td><a href="#prompt-${i + 1}" style="text-decoration: none; color: inherit;">${escapeHTML(promptPreview)}</a></td>
              </tr>`;
        }).join("");

        indexContainer.innerHTML = `
            <h2 style="margin-top:0; border-bottom: 1px solid var(--border-color); padding-bottom: 5px;">Table of Contents</h2>
            <div class="box-content-html" style="margin: 0;">
                <table style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr>
                            <th style="width: 60px; text-align: center;">Sr No.</th>
                            <th style="text-align: left;">Prompt Details</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${indexRows}
                    </tbody>
                </table>
            </div>
        `;
        resultsContainer.appendChild(indexContainer);

        results.forEach((res, index) => {
            const item = document.createElement("div");
            item.id = `prompt-${index + 1}`; // Match link in index
            item.className = "result-item";

            const promptDiv = document.createElement("div");
            promptDiv.className = "prompt-box";
            promptDiv.innerHTML = `<div class="box-label">Prompt ${index + 1}:</div><div class="box-content">${escapeHTML(res.prompt)}</div>`;

            const responseDiv = document.createElement("div");
            responseDiv.className = "response-box";
            const contentHtml = res.responseHtml || escapeHTML(res.response || "No response extracted.");
            responseDiv.innerHTML = `<div class="box-label">Response:</div><div class="box-content-html">${contentHtml}</div>`;

            item.appendChild(promptDiv);
            item.appendChild(responseDiv);
            resultsContainer.appendChild(item);
        });
    }

    function escapeHTML(str) {
        if (!str) return "";
        return str.replace(/[&<>'"]/g,
            tag => ({
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                "'": '&#39;',
                '"': '&quot;'
            }[tag] || tag)
        ).replace(/\n/g, '<br>');
    }
});
