document.addEventListener("DOMContentLoaded", () => {
    const historyList = document.getElementById("history-list");
    const clearAllBtn = document.getElementById("clear-all-btn");

    // Setup theme
    const savedTheme = localStorage.getItem("theme");
    if (savedTheme) {
        document.documentElement.setAttribute("data-theme", savedTheme);
    }

    loadHistory();

    clearAllBtn.addEventListener("click", () => {
        if (confirm("Are you sure you want to clear all history? This cannot be undone.")) {
            chrome.storage.local.set({ historyLogs: [] }, () => {
                loadHistory();
            });
        }
    });

    function loadHistory() {
        chrome.storage.local.get(["historyLogs"], (result) => {
            const logs = result.historyLogs || [];
            renderHistory(logs);
        });
    }

    function renderHistory(logs) {
        historyList.innerHTML = "";
        if (logs.length === 0) {
            historyList.innerHTML = `<div class="no-history">No history found. Start a queue to see results here!</div>`;
            return;
        }

        logs.forEach((log) => {
            const card = document.createElement("div");
            card.className = "history-card";

            const isComplete = log.completedPrompts >= log.totalPrompts;
            const statusText = isComplete ? "Completed" : "Stopped Partial";

            card.innerHTML = `
        <div class="history-card-header">
          <div>
            <div class="history-card-title">${log.date}</div>
            <div class="history-card-meta">
              Prompts: ${log.completedPrompts} / ${log.totalPrompts} &middot; ${statusText}
            </div>
          </div>
          <div class="history-card-actions">
            <button class="btn secondary toggle-details-btn"><span class="material-symbols-outlined">visibility</span> View Details</button>
            <button class="btn secondary export-json-btn" title="Export JSON"><span class="material-symbols-outlined">data_object</span> JSON</button>
            <button class="btn secondary export-csv-btn" title="Export CSV"><span class="material-symbols-outlined">grid_on</span> CSV</button>
            <button class="btn secondary export-html-btn" title="Export HTML"><span class="material-symbols-outlined">html</span> HTML</button>
            <button class="btn danger delete-btn" title="Delete from history"><span class="material-symbols-outlined">delete</span> Delete</button>
          </div>
        </div>
        <div class="history-details">
          ${log.results.map((res, i) => `
            <div class="prompt-item">
              <div class="prompt-label">Prompt ${i + 1}</div>
              <div class="prompt-text">${escapeHTML(res.prompt)}</div>
              <div class="response-label">Response</div>
              <div class="response-text">${res.responseHtml || escapeHTML(res.response)}</div>
            </div>
          `).join('')}
        </div>
      `;

            // Setup actions
            const toggleDetailsBtn = card.querySelector(".toggle-details-btn");
            const detailsContainer = card.querySelector(".history-details");
            toggleDetailsBtn.addEventListener("click", () => {
                const isOpen = detailsContainer.classList.contains("open");
                if (isOpen) {
                    detailsContainer.classList.remove("open");
                    toggleDetailsBtn.innerHTML = '<span class="material-symbols-outlined">visibility</span> View Details';
                } else {
                    detailsContainer.classList.add("open");
                    toggleDetailsBtn.innerHTML = '<span class="material-symbols-outlined">visibility_off</span> Hide Details';
                }
            });

            card.querySelector(".delete-btn").addEventListener("click", () => {
                if (confirm("Delete this session from history?")) {
                    deleteLog(log.id);
                }
            });

            card.querySelector(".export-json-btn").addEventListener("click", () => {
                exportJson(log);
            });

            card.querySelector(".export-csv-btn").addEventListener("click", () => {
                exportCsv(log);
            });

            card.querySelector(".export-html-btn").addEventListener("click", () => {
                exportHtml(log);
            });

            historyList.appendChild(card);
        });
    }

    function deleteLog(id) {
        chrome.storage.local.get(["historyLogs"], (result) => {
            let logs = result.historyLogs || [];
            logs = logs.filter((log) => log.id !== id);
            chrome.storage.local.set({ historyLogs: logs }, () => {
                loadHistory();
            });
        });
    }

    // Common export utils
    function downloadBlob(content, filename, contentType) {
        const blob = new Blob([content], { type: contentType });
        const url = URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.href = url;
        a.download = filename;
        a.click();
        URL.revokeObjectURL(url);
    }

    function escapeHTML(str) {
        if (!str) return "";
        return str
            .replace(/[&<>'"]/g, (tag) => ({ "&": "&amp;", "<": "&lt;", ">": "&gt;", "'": "&#39;", '"': "&quot;" })[tag] || tag)
            .replace(/\n/g, "<br>");
    }

    function exportJson(log) {
        const jsonContent = JSON.stringify(log.results, null, 2);
        downloadBlob(jsonContent, `ai_autopilot_history_${log.id}.json`, "application/json");
    }

    function exportCsv(log) {
        let csvContent = "\uFEFFPrompt,Response\n"; // Include BOM for UTF-8
        log.results.forEach((res) => {
            let p = (res.prompt || "").replace(/"/g, '""');
            let r = (res.response || "").replace(/"/g, '""');
            csvContent += `"${p}","${r}"\n`;
        });
        downloadBlob(csvContent, `ai_autopilot_history_${log.id}.csv`, "text/csv;charset=utf-8;");
    }

    function exportHtml(log) {
        let htmlBlocks = log.results
            .map((res, index) => {
                const contentHtml = res.responseHtml || escapeHTML(res.response || "No response extracted.");
                return `
            <div style="margin-bottom: 20px; border: 1px solid #ccc; padding: 15px; border-radius: 8px; page-break-inside: avoid;">
                <h3 style="margin-top:0; color: #333; border-bottom: 1px solid #eee; padding-bottom: 5px;">Prompt ${index + 1}</h3>
                <p style="background-color: #f9f9f9; padding: 10px; border-radius: 4px; white-space: pre-wrap; font-family: sans-serif;">${escapeHTML(res.prompt)}</p>
                <div class="ai-content" style="margin-top:15px; color: #333; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6;">
                    ${contentHtml}
                </div>
            </div>`;
            })
            .join("");

        const fullHtml = `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>AI Auto-Pilot History - ${log.date}</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 800px; margin: 20px auto; padding: 0 20px; line-height: 1.6; color: #333; }
        h1 { border-bottom: 2px solid #0056b3; padding-bottom: 10px; color: #0056b3; }
        .ai-content table { width: 100%; border-collapse: collapse; margin: 15px 0; font-size: 15px; }
        .ai-content th, .ai-content td { border: 1px solid #ddd; padding: 10px; text-align: left; }
        .ai-content th { background-color: #f2f2f2; font-weight: bold; }
        .ai-content tr:nth-child(even) { background-color: #f9f9f9; }
    </style>
</head>
<body>
    <h1>AI Auto-Pilot Export: ${log.date}</h1>
    <p>Status: ${log.completedPrompts} / ${log.totalPrompts}</p>
    ${htmlBlocks}
</body>
</html>`;

        downloadBlob("\uFEFF" + fullHtml, `ai_autopilot_history_${log.id}.html`, "text/html;charset=utf-8;");
    }
});
