document.addEventListener("DOMContentLoaded", () => {
  // UI Elements
  const themeToggle = document.getElementById("theme-toggle");
  const promptQueueTextArea = document.getElementById("prompt-queue");
  const delayTimeInput = document.getElementById("delay-time");
  const startBtn = document.getElementById("start-btn");
  const pauseBtn = document.getElementById("pause-btn");
  const stopBtn = document.getElementById("stop-btn");
  const historyBtn = document.getElementById("history-btn");
  const landingBtn = document.getElementById("landing-btn");
  const progressBar = document.getElementById("progress-bar");
  const statusLabel = document.getElementById("status-label");
  const progressCount = document.getElementById("progress-count");
  const smartRetryToggle = document.getElementById("smart-retry");

  // Template UI Elements
  const templateSelect = document.getElementById("template-select");
  const loadTemplateBtn = document.getElementById("load-template-btn");
  const saveTemplateBtn = document.getElementById("save-template-btn");
  const deleteTemplateBtn = document.getElementById("delete-template-btn");

  // Export UI Elements
  const exportActions = document.getElementById("export-actions");
  const exportJsonBtn = document.getElementById("export-json");
  const exportCsvBtn = document.getElementById("export-csv");
  const exportPdfBtn = document.getElementById("export-pdf");
  const exportHtmlBtn = document.getElementById("export-html");
  const exportDocBtn = document.getElementById("export-doc");
  const exportCustomTitle = document.getElementById("export-custom-title");

  // Load theme
  const savedTheme = localStorage.getItem("theme");
  const themeIcon = document.getElementById("theme-icon");
  if (savedTheme) {
    document.documentElement.setAttribute("data-theme", savedTheme);
    if (themeIcon) {
      themeIcon.textContent = savedTheme === "dark" ? "light_mode" : "dark_mode";
    }
  }

  themeToggle.addEventListener("click", () => {
    const currentTheme = document.documentElement.getAttribute("data-theme");
    const newTheme = currentTheme === "dark" ? "light" : "dark";
    document.documentElement.setAttribute("data-theme", newTheme);
    localStorage.setItem("theme", newTheme);
    if (themeIcon) {
      themeIcon.textContent = newTheme === "dark" ? "light_mode" : "dark_mode";
    }
  });

  // Window Controls
  const minimizeBtn = document.getElementById("minimize-btn");
  const closePopupBtn = document.getElementById("close-popup-btn");
  const mainContent = document.querySelector(".main-content");

  if (closePopupBtn) {
    closePopupBtn.addEventListener("click", () => {
      window.close();
    });
  }



  if (minimizeBtn) {
    minimizeBtn.addEventListener("click", () => {
      mainContent.classList.toggle("minimized");
      const icon = minimizeBtn.querySelector("span");
      if (mainContent.classList.contains("minimized")) {
        icon.textContent = "expand_more";
        minimizeBtn.title = "Expand";
      } else {
        icon.textContent = "remove";
        minimizeBtn.title = "Minimize";
      }
    });
  }

  // Toast Notification System
  function showToast(message, type = "info", duration = 3000) {
    const container = document.getElementById("toast-container");
    if (!container) return;

    const toast = document.createElement("div");
    toast.className = `toast ${type}`;

    let icon = "info";
    if (type === "success") icon = "check_circle";
    if (type === "error") icon = "error";

    toast.innerHTML = `
      <span class="material-symbols-outlined toast-icon">${icon}</span>
      <span class="toast-message">${message}</span>
    `;

    container.appendChild(toast);

    // Trigger animation
    requestAnimationFrame(() => {
      toast.classList.add("show");
    });

    setTimeout(() => {
      toast.classList.remove("show");
      setTimeout(() => toast.remove(), 400); // Wait for transition
    }, duration);
  }

  // Function to update UI from state
  function updateUI(state) {
    if (!state) return;

    // Update inputs state
    promptQueueTextArea.disabled = state.isRunning || state.isPaused;
    delayTimeInput.disabled = state.isRunning || state.isPaused;

    if (state.smartRetry !== undefined && smartRetryToggle) {
      smartRetryToggle.checked = state.smartRetry;
    }

    // If not running and queue has prompts from previous session, restore it
    if (
      !state.isRunning &&
      !state.isPaused &&
      state.queue.length > 0 &&
      promptQueueTextArea.value === ""
    ) {
      promptQueueTextArea.value = state.queue.join("\n\n");
    }

    const total = state.totalPrompts || 0;
    const current = state.currentIndex || 0;

    progressCount.textContent = `${current} / ${total}`;
    const percentage = total > 0 ? (current / total) * 100 : 0;
    progressBar.style.width = `${percentage}%`;

    if (state.isRunning) {
      statusLabel.textContent = "Status: Running...";
      statusLabel.style.color = "var(--primary)";
      startBtn.disabled = true;
      pauseBtn.disabled = false;
      stopBtn.disabled = false;
      pauseBtn.innerHTML = '<span class="material-symbols-outlined">pause</span> Pause';
    } else if (state.isPaused) {
      statusLabel.textContent = "Status: Paused";
      statusLabel.style.color = "var(--text-secondary)";
      startBtn.disabled = true;
      pauseBtn.disabled = false;
      stopBtn.disabled = false;
      pauseBtn.innerHTML = '<span class="material-symbols-outlined">play_arrow</span> Resume';
    } else {
      statusLabel.textContent = "Status: Idle";
      statusLabel.style.color = "var(--text-secondary)";
      startBtn.disabled = false;
      pauseBtn.disabled = true;
      stopBtn.disabled = true;
    }

    // Handle export buttons visibility
    if (state.results && state.results.length > 0) {
      exportActions.classList.add("visible");
    } else {
      exportActions.classList.remove("visible");
    }
  }

  // Request initial state from background script
  chrome.runtime.sendMessage({ type: "GET_STATE" }, (response) => {
    if (response) updateUI(response);
  });

  // Listen for state updates broadcasted by background script
  chrome.runtime.onMessage.addListener((message) => {
    if (message.type === "STATE_UPDATE") {
      updateUI(message.state);
    } else if (message.type === "QUEUE_COMPLETED") {
      playCompletionSound();
      showToast(`🎉 AI Auto-Pilot finished processing ${message.total} prompts successfully!`, "success", 5000);

      // Force UI refresh simply by requesting state again
      chrome.runtime.sendMessage({ type: "GET_STATE" }, (response) => {
        if (response) updateUI(response);
      });
    }
  });

  function playCompletionSound() {
    // Create an oscillator to generate a pleasant non-obtrusive "ding" sound
    try {
      const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
      const oscillator = audioCtx.createOscillator();
      const gainNode = audioCtx.createGain();

      oscillator.type = 'sine';
      oscillator.frequency.setValueAtTime(523.25, audioCtx.currentTime); // C5
      oscillator.frequency.exponentialRampToValueAtTime(1046.50, audioCtx.currentTime + 0.1); // C6

      gainNode.gain.setValueAtTime(0, audioCtx.currentTime);
      gainNode.gain.linearRampToValueAtTime(0.5, audioCtx.currentTime + 0.05);
      gainNode.gain.exponentialRampToValueAtTime(0.01, audioCtx.currentTime + 1);

      oscillator.connect(gainNode);
      gainNode.connect(audioCtx.destination);

      oscillator.start();
      oscillator.stop(audioCtx.currentTime + 1);
    } catch (e) {
      console.log("AudioContext not supported or blocked by browser policy.");
    }
  }

  if (smartRetryToggle) {
    smartRetryToggle.addEventListener("change", (e) => {
      chrome.runtime.sendMessage({
        type: "UPDATE_SETTINGS",
        payload: { smartRetry: e.target.checked },
      });
    });
  }

  // Button Listeners
  startBtn.addEventListener("click", () => {
    const text = promptQueueTextArea.value.trim();
    if (!text) {
      showToast("Please enter some prompts.", "error");
      return;
    }

    // Split by double newline to handle multiline prompts easily
    let prompts = text
      .split(/\n\s*\n/)
      .map((p) => p.trim())
      .filter((p) => p);

    // Fallback: If there are no double newlines, but there are multiple lines, assume they meant 1 line = 1 prompt.
    if (prompts.length === 1 && text.includes("\n")) {
      prompts = text
        .split("\n")
        .map((p) => p.trim())
        .filter((p) => p);
    }

    if (prompts.length > 10) {
      showToast("Free version is limited to 10 prompts. Get Premium for unlimited prompts!", "error", 5500);
      prompts = prompts.slice(0, 10);
      promptQueueTextArea.value = prompts.join("\n\n");
    }

    const delay = parseInt(delayTimeInput.value) || 5;
    const smartRetry = false; // Disabled in Free version

    chrome.runtime.sendMessage({
      type: "START",
      payload: { prompts, delay, smartRetry },
    });
  });

  pauseBtn.addEventListener("click", () => {
    chrome.runtime.sendMessage({ type: "TOGGLE_PAUSE" });
  });

  stopBtn.addEventListener("click", () => {
    chrome.runtime.sendMessage({ type: "STOP" });
    promptQueueTextArea.value = "";
    progressBar.style.width = "0%";
    progressCount.textContent = "0 / 0";
  });

  if (historyBtn) {
    historyBtn.addEventListener("click", () => {
      chrome.tabs.create({ url: "history.html" });
    });
  }

  if (landingBtn) {
    landingBtn.addEventListener("click", () => {
      chrome.tabs.create({ url: "landing.html" });
    });
  }

  // --- Workflow Templates Functionality ---

  // Load templates from storage and render dropdown
  function loadTemplates() {
    chrome.storage.local.get("workflowTemplates", (result) => {
      const templates = result.workflowTemplates || {};
      // Clear existing options except the first one
      templateSelect.innerHTML =
        '<option value="">-- Select a Template --</option>';

      for (const name in templates) {
        const option = document.createElement("option");
        option.value = name;
        option.textContent = name;
        templateSelect.appendChild(option);
      }

      // Disable delete and load if no template is selected
      deleteTemplateBtn.disabled = true;
      loadTemplateBtn.disabled = true;
    });
  }

  templateSelect.addEventListener("change", (e) => {
    const hasSelection = e.target.value !== "";
    deleteTemplateBtn.disabled = !hasSelection;
    loadTemplateBtn.disabled = !hasSelection;
  });

  saveTemplateBtn.addEventListener("click", () => {
    const currentPrompts = promptQueueTextArea.value.trim();
    if (!currentPrompts) {
      showToast("No prompts to save. Please enter some prompts first.", "error", 4000);
      return;
    }

    const templateName = prompt(
      "Enter a name for this template:",
      "My Template",
    );
    if (!templateName) return; // User cancelled

    chrome.storage.local.get("workflowTemplates", (result) => {
      const templates = result.workflowTemplates || {};
      templates[templateName] = currentPrompts;
      chrome.storage.local.set({ workflowTemplates: templates }, () => {
        showToast(`Template "${templateName}" saved successfully!`, "success");
        loadTemplates();
        // Select the newly saved template
        setTimeout(() => {
          templateSelect.value = templateName;
          templateSelect.dispatchEvent(new Event("change"));
        }, 100);
      });
    });
  });

  loadTemplateBtn.addEventListener("click", () => {
    const selectedName = templateSelect.value;
    if (!selectedName) return;

    chrome.storage.local.get("workflowTemplates", (result) => {
      const templates = result.workflowTemplates || {};
      if (templates[selectedName]) {
        promptQueueTextArea.value = templates[selectedName];
      } else {
        showToast("Template not found.", "error");
      }
    });
  });

  deleteTemplateBtn.addEventListener("click", () => {
    const selectedName = templateSelect.value;
    if (!selectedName) return;

    if (
      !confirm(
        `Are you sure you want to delete the template "${selectedName}"?`,
      )
    )
      return;

    chrome.storage.local.get("workflowTemplates", (result) => {
      const templates = result.workflowTemplates || {};
      if (templates[selectedName]) {
        delete templates[selectedName];
        chrome.storage.local.set({ workflowTemplates: templates }, () => {
          showToast(`Template "${selectedName}" deleted.`, "info");
          loadTemplates();
        });
      }
    });
  });

  // Initialize templates on load
  loadTemplates();

  // --- Export Functionality ---

  function downloadBlob(content, filename, contentType) {
    const blob = new Blob([content], { type: contentType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
  }

  exportJsonBtn.addEventListener("click", () => {
    chrome.runtime.sendMessage({ type: "GET_STATE" }, (state) => {
      if (state && state.results && state.results.length > 0) {
        const jsonContent = JSON.stringify(state.results, null, 2);
        let safeFilename = getExportTitle().replace(/[^a-z0-9]/gi, '_').toLowerCase();
        downloadBlob(
          jsonContent,
          `${safeFilename}_results.json`,
          "application/json",
        );
      } else {
        showToast("No results to export yet.", "error");
      }
    });
  });

  exportCsvBtn.addEventListener("click", () => {
    chrome.runtime.sendMessage({ type: "GET_STATE" }, (state) => {
      if (state && state.results && state.results.length > 0) {
        let csvContent = "\uFEFFPrompt,Response\n"; // Include BOM for UTF-8
        state.results.forEach((res) => {
          let p = (res.prompt || "").replace(/"/g, '""');
          let r = (res.response || "").replace(/"/g, '""');
          csvContent += `"${p}","${r}"\n`;
        });
        let safeFilename = getExportTitle().replace(/[^a-z0-9]/gi, '_').toLowerCase();
        downloadBlob(
          csvContent,
          `${safeFilename}_results.csv`,
          "text/csv;charset=utf-8;",
        );
      } else {
        showToast("No results to export yet.", "error");
      }
    });
  });

  // Replaced with premium-feature-lock click below

  function getExportTitle() {
    return exportCustomTitle && exportCustomTitle.value.trim() !== ""
      ? exportCustomTitle.value.trim()
      : "AI Auto-Pilot Export";
  }

  function generateHtmlContent(results) {
    const title = getExportTitle();

    // Generate Index Table
    let indexRows = results.map((res, i) => {
      // Create a short preview of the prompt for the index
      let promptPreview = res.prompt || `Response ${i + 1}`;
      if (promptPreview.length > 60) {
        promptPreview = promptPreview.substring(0, 60) + "...";
      }
      return `
        <tr>
          <td style="text-align: center;">${i + 1}</td>
          <td><a href="#prompt-${i + 1}" style="text-decoration: none; color: #0056b3;">${escapeHTML(promptPreview)}</a></td>
        </tr>`;
    }).join("");

    let indexHtml = `
      <div style="margin-bottom: 30px; border: 1px solid #ccc; padding: 15px; border-radius: 8px; page-break-after: always;">
        <h2 style="margin-top:0; color: #333; border-bottom: 1px solid #eee; padding-bottom: 5px;">Table of Contents</h2>
        <table style="width: 100%; border-collapse: collapse; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;">
          <thead>
            <tr>
              <th style="border: 1px solid #ddd; padding: 10px; background-color: #f2f2f2; width: 60px; text-align: center;">Sr No.</th>
              <th style="border: 1px solid #ddd; padding: 10px; background-color: #f2f2f2; text-align: left;">Prompt Details</th>
            </tr>
          </thead>
          <tbody>
            ${indexRows}
          </tbody>
        </table>
      </div>
    `;

    let htmlBlocks = results
      .map((res, index) => {
        const contentHtml =
          res.responseHtml ||
          escapeHTML(res.response || "No response extracted.");
        return `
            <div id="prompt-${index + 1}" style="margin-bottom: 20px; border: 1px solid #ccc; padding: 15px; border-radius: 8px; page-break-inside: avoid;">
                <h3 style="margin-top:0; color: #333; border-bottom: 1px solid #eee; padding-bottom: 5px;">Prompt ${index + 1}</h3>
                <p style="background-color: #f9f9f9; padding: 10px; border-radius: 4px; white-space: pre-wrap; font-family: sans-serif;">${escapeHTML(res.prompt)}</p>
                <div class="ai-content" style="margin-top:15px; color: #333; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; line-height: 1.6;">
                    ${contentHtml}
                </div>
            </div>`;
      })
      .join("");

    return `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>${escapeHTML(title)}</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; max-width: 800px; margin: 20px auto; padding: 0 20px; line-height: 1.6; color: #333; }
        h1 { border-bottom: 2px solid #0056b3; padding-bottom: 10px; color: #0056b3; }
        a:hover { text-decoration: underline !important; }
        .ai-content { overflow-wrap: break-word; }
        .ai-content table { width: 100%; border-collapse: collapse; margin: 15px 0; font-size: 15px; page-break-inside: auto; }
        .ai-content tr { page-break-inside: avoid; page-break-after: auto; }
        .ai-content thead { display: table-header-group; }
        .ai-content th, .ai-content td { border: 1px solid #ddd; padding: 10px; text-align: left; }
        .ai-content th { background-color: #f2f2f2; font-weight: bold; }
        .ai-content tr:nth-child(even) { background-color: #f9f9f9; }
        .ai-content div, .ai-content pre { overflow: visible !important; white-space: pre-wrap !important; }
    </style>
</head>
<body>
    <h1>${escapeHTML(title)}</h1>
    <p>Exported on: ${new Date().toLocaleString()}</p>
    ${indexHtml}
    ${htmlBlocks}
</body>
</html>`;
  }

  function escapeHTML(str) {
    if (!str) return "";
    return str
      .replace(
        /[&<>'"]/g,
        (tag) =>
          ({
            "&": "&amp;",
            "<": "&lt;",
            ">": "&gt;",
            "'": "&#39;",
            '"': "&quot;",
          })[tag] || tag,
      )
      .replace(/\n/g, "<br>");
  }

  exportHtmlBtn.addEventListener("click", () => {
    chrome.runtime.sendMessage({ type: "GET_STATE" }, (state) => {
      if (state && state.results && state.results.length > 0) {
        const htmlContent = "\uFEFF" + generateHtmlContent(state.results);
        let safeFilename = getExportTitle().replace(/[^a-z0-9]/gi, '_').toLowerCase();
        downloadBlob(
          htmlContent,
          `${safeFilename}_results.html`,
          "text/html;charset=utf-8;",
        );
      } else {
        showToast("No results to export yet.", "error");
      }
    });
  });

  // Replaced with premium-feature-lock click below

  // Premium features redirect
  const premiumLocks = document.querySelectorAll(".premium-feature-lock");
  premiumLocks.forEach(lock => {
    lock.addEventListener("click", (e) => {
      // Prevent other click events from firing (like checkbox toggle)
      e.preventDefault();
      chrome.tabs.create({ url: "landing.html" });
    });
  });
});
